<?php
require_once __DIR__ . '/../config/db.php';

$pdo = getDBConnection();
$message = '';
$messageType = '';

// Handle board template addition
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_board'])) {
    $name = trim($_POST['board_name']);
    $description = trim($_POST['board_description']);
    $categoryId = (int)$_POST['category_id'];
    
    if (!empty($name) && $categoryId > 0) {
        try {
            $stmt = $pdo->prepare("INSERT INTO board_data (name, description, category_id) VALUES (:name, :description, :category_id)");
            $stmt->execute([
                'name' => $name,
                'description' => $description,
                'category_id' => $categoryId
            ]);
            $message = "Board template added successfully!";
            $messageType = "success";
        } catch (PDOException $e) {
            $message = "Error adding board: " . $e->getMessage();
            $messageType = "error";
        }
    }
}

// Get all categories for dropdown
$categoriesStmt = $pdo->query("SELECT id, category_name FROM categories ORDER BY category_name");
$categories = $categoriesStmt->fetchAll();

// Get all board templates
$boardsStmt = $pdo->query("
    SELECT bd.*, c.category_name,
    COUNT(ab.id) as usage_count
    FROM board_data bd
    LEFT JOIN categories c ON bd.category_id = c.id
    LEFT JOIN account_board ab ON bd.bid = ab.bid
    GROUP BY bd.bid
    ORDER BY c.category_name, bd.name
");
$boards = $boardsStmt->fetchAll();

// Get account boards
$accountBoardsStmt = $pdo->query("
    SELECT ab.*, bd.name as board_name, c.category_name
    FROM account_board ab
    LEFT JOIN board_data bd ON ab.bid = bd.bid
    LEFT JOIN categories c ON bd.category_id = c.id
    ORDER BY ab.creation_date DESC
    LIMIT 50
");
$accountBoards = $accountBoardsStmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Boards - Pinterest Pin Manager</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: #333;
            font-size: 2em;
            margin-bottom: 10px;
        }
        
        .back-link {
            display: inline-block;
            color: #667eea;
            text-decoration: none;
            margin-top: 10px;
            font-size: 0.9em;
        }
        
        .message {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
        }
        
        .add-form {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .add-form h2 {
            color: #333;
            margin-bottom: 15px;
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
        }
        
        .form-group.full {
            grid-column: 1 / -1;
        }
        
        .form-group label {
            margin-bottom: 5px;
            font-size: 0.9em;
            color: #666;
            font-weight: 500;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 1em;
        }
        
        .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1em;
        }
        
        .section {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .section h2 {
            color: #333;
            margin-bottom: 20px;
            font-size: 1.5em;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        thead {
            background: #f8f9fa;
        }
        
        th {
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #e0e0e0;
        }
        
        td {
            padding: 12px;
            border-bottom: 1px solid #f0f0f0;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85em;
            background: #e7f3ff;
            color: #0066cc;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px;
            color: #999;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📊 Board Management</h1>
            <a href="index.php" class="back-link">← Back to Dashboard</a>
        </div>
        
        <?php if ($message): ?>
            <div class="message <?php echo $messageType; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>
        
        <div class="add-form">
            <h2>Add Board Template</h2>
            <form method="POST">
                <div class="form-grid">
                    <div class="form-group">
                        <label>Board Name *</label>
                        <input type="text" name="board_name" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Category *</label>
                        <select name="category_id" required>
                            <option value="">Select Category</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>">
                                    <?php echo htmlspecialchars($cat['category_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group full">
                        <label>Description</label>
                        <textarea name="board_description"></textarea>
                    </div>
                </div>
                
                <button type="submit" name="add_board" class="btn">Add Board Template</button>
            </form>
        </div>
        
        <div class="section">
            <h2>Board Templates</h2>
            <?php if (empty($boards)): ?>
                <div class="empty-state">No board templates yet</div>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Board Name</th>
                            <th>Category</th>
                            <th>Description</th>
                            <th>Usage</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($boards as $board): ?>
                            <tr>
                                <td><?php echo $board['bid']; ?></td>
                                <td><strong><?php echo htmlspecialchars($board['name']); ?></strong></td>
                                <td><span class="badge"><?php echo htmlspecialchars($board['category_name']); ?></span></td>
                                <td><?php echo htmlspecialchars(substr($board['description'], 0, 100)); ?></td>
                                <td><?php echo $board['usage_count']; ?> account(s)</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <div class="section">
            <h2>Account Boards (Recent)</h2>
            <?php if (empty($accountBoards)): ?>
                <div class="empty-state">No account boards registered yet</div>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>Account</th>
                            <th>Board Name</th>
                            <th>Category</th>
                            <th>Pinterest Board ID</th>
                            <th>Created</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($accountBoards as $ab): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($ab['acc_username']); ?></strong></td>
                                <td><?php echo htmlspecialchars($ab['board_name']); ?></td>
                                <td><span class="badge"><?php echo htmlspecialchars($ab['category_name']); ?></span></td>
                                <td><code><?php echo htmlspecialchars($ab['bord_id']); ?></code></td>
                                <td><?php echo date('M d, Y H:i', strtotime($ab['creation_date'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
